/*
 * @brief Image programming code
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */
 #include <string.h>
 #include "sl_common.h"
 #include "sl_flash.h"
 #include "fsl_iap.h"
 
 #define ERR_ISP_INVALID_SECTOR 7
/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
volatile int isSecure = 0;
volatile uint32_t keyFlash[SL_KEY_BYTES / sizeof(uint32_t)];
/*****************************************************************************
 * Private functions
 ****************************************************************************/

/*****************************************************************************
 * Public functions
 ****************************************************************************/
uint32_t GetSecNum (uint32_t adr)
{
    uint32_t n;
    n = adr >> 10;                               // 1kB Sector (0x400 each)
    return (n);                                  // Sector Number
}

int32_t flashWriteBlock(uint32_t *src, uint32_t block_nr)
{
    uint32_t dst, start_sect, rc;
    uint32_t adr=0;

    dst = (SL_FLASH_BLOCK_SZ * block_nr);
    if ( (dst < SL_BOOTAPP_ADDR1) || ((dst + SL_FLASH_BLOCK_SZ) > SL_FLASH_END )) {
        return ERR_API_INVALID_PARAMS;
    }
    start_sect = (dst / SL_FLASH_SECT_SZ);

    /* Erase sector when write address is on sector boundary */
    if ( (dst % SL_FLASH_SECT_SZ) == 0x0) {
        __disable_irq();

		IAP_PrepareSectorForWrite(start_sect, start_sect);
		rc = IAP_EraseSector(start_sect, start_sect, SystemCoreClock);
        
        __enable_irq();
        if (rc != LPC_OK) {
            return rc;
        }
    }
    __disable_irq();
        
	IAP_PrepareSectorForWrite(start_sect, start_sect);
	rc = IAP_CopyRamToFlash(dst, src, SL_FLASH_BLOCK_SZ, SystemCoreClock);
    
    __enable_irq();

    return rc;
}

int32_t flashErasePage(uint32_t start_addr, uint32_t end_addr)
{
    uint32_t rc;
	uint32_t sec;
	uint32_t page;
    if ( (start_addr < SL_BOOTAPP_ADDR1) ) {
        /* Protect secondary loader */
        return ERR_ISP_INVALID_SECTOR;
    }
    if ( (end_addr < SL_BOOTAPP_ADDR1) ) {
        /* Protect secondary loader */
        return ERR_ISP_INVALID_SECTOR;
    }
    if ( (start_addr % SL_FLASH_PAGE_SZ) != 0 ) {
        return ERR_ISP_DST_ADDR_ERROR;
    }
    if ( (end_addr % SL_FLASH_PAGE_SZ) != 0 ) {
        return ERR_ISP_DST_ADDR_ERROR;
    }

    __disable_irq();
    
	sec = GetSecNum (start_addr);
	page = start_addr / SL_FLASH_PAGE_SZ;
    /* Erase page */
    IAP_PrepareSectorForWrite(sec, sec);
    rc = IAP_ErasePage(page, page, SystemCoreClock);
	
    __enable_irq();
    return rc;
}

int32_t flashWritePage(uint32_t dst, uint32_t *src)
{
    uint32_t rc;
	uint32_t sec;

    if ( (dst  < SL_BOOTAPP_ADDR1) ) {
        /* Protect secondary loader */
        return ERR_ISP_INVALID_SECTOR;
    }
    if ( (dst % SL_FLASH_PAGE_SZ) != 0 ) {
        return ERR_ISP_DST_ADDR_ERROR;
    }

    __disable_irq();
    
	sec = GetSecNum (dst);
	IAP_PrepareSectorForWrite(sec, sec);
	rc = IAP_CopyRamToFlash(dst, src, SL_FLASH_PAGE_SZ, SystemCoreClock);
	
    __enable_irq();
    return rc;
}

int32_t flashEraseSector(uint32_t start_sect, uint32_t end_sect)
{
    uint32_t rc;
    if ( (start_sect == 0) || (end_sect == 0)) {
        /* Protect secondary loader */
        return ERR_ISP_INVALID_SECTOR;
    }
    __disable_irq();
    
    IAP_PrepareSectorForWrite(start_sect, end_sect);
    rc = IAP_EraseSector(start_sect, end_sect, SystemCoreClock);
    
    __enable_irq();
    return rc;
}

int32_t flashReadBlock(uint32_t *dest, uint32_t block_nr)
{
    uint32_t src = (SL_FLASH_BLOCK_SZ * block_nr);

    if ( (src + SL_FLASH_BLOCK_SZ) > SL_FLASH_END ) {
        return ERR_API_INVALID_PARAMS;
    }
    memcpy(dest, (void *) src, SL_FLASH_BLOCK_SZ);

    return LPC_OK;
}

int32_t flashReadPage(uint32_t *dest, uint32_t page_nr)
{
    uint32_t src = (SL_FLASH_PAGE_SZ * page_nr);

    if ( (src + SL_FLASH_PAGE_SZ) > SL_FLASH_END ) {
        return ERR_API_INVALID_PARAMS;
    }
    memcpy(dest, (void *) src, SL_FLASH_PAGE_SZ);

    return LPC_OK;
}
